<?php
namespace App\Http\Controllers\Api\Driver;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Services\ParcelService;
use App\Http\Requests\ParcelRequst;
use Illuminate\Support\Facades\Storage;

use App\Models\Parcels;
use Carbon\Carbon;


class ParcelController extends Controller
{
	protected $parcelService;

	public function __construct(ParcelService $parcelService)
	{
		$this->parcelService = $parcelService;
	}

    public function singleMode(ParcelRequst $request)
    {
        $data = $request->validated();
        $data['driver_id'] = $request->login_detail['login_id'];
        $parcel = $this->parcelService->createParcel($data);

        return apiResponse(1, 'Single Mode ride created successfully.', $parcel, 200);
    }		

    public function checkNewParcel(Request $request)
    {
        $latestParcel = $this->parcelService->getLatestParcel();
        if($latestParcel)
        return apiResponse(1, 'Latest Parcel', $latestParcel, 200);
        else
        return apiResponse(0, 'There is no new parcel', [], 200);
    }	

	public function getPendingParcels(Request $request)
	{
		$login_detail 	= $request->login_detail;	
		$driver_login_id= $login_detail['login_id'];

		$parcels = $this->parcelService->getPendingParcelsForDriver($driver_login_id);
        
		return apiResponse(1, 'Pending Parcels', $parcels, 200);	
	}

    public function getNearbyParcels(ParcelRequst $request)
    {
        $login_detail   = $request->login_detail;   
        $driver_login_id= $login_detail['login_id'];

        $pendingRides = Parcels::with(['user:id,name,mobile,photo'])
            ->whereIn('status', [PENDING, ASSIGNED])
            ->whereNotIn('id', function ($query) use ($driver_login_id) {
                $query->select('parcel_id')
                ->from('parcel_requests')
                ->where('status', '2')
                ->where('driver_id', $driver_login_id);
            })
            ->orderby('id', 'DESC')
            ->get(['id', 'user_id', 'pickup_address', 'delivery_address', 'final_price', 'status']);
        
        return apiResponse(1, '', $pendingRides, 200);

        //comment for the timing

        $validated = $request->validated();
        $parcels = $this->parcelService->getNearbyParcels(
            $login_detail,
            $validated['status'] ?? '1',
            $validated['latitude'],
            $validated['longitude'],
            $validated['radius'] ?? 10
        );

        return apiResponse(1, 'Near By Parcels', $parcels, 200);
    }

    public function rejectParcel(Request $request)
    {
        $login_detail   = $request->login_detail;   
        $driverId = $login_detail['login_id'];


        if ($this->parcelService->rejectParcel($request->parcel_id, $driverId)) {
            return apiResponse(1, 'Parcel rejected successfully.', '', 200);
        }

        return apiResponse(0, 'Failed to reject parcel.', '', 400);
    }

    public function acceptParcel(ParcelRequst $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];
        $is_schedule = $request->is_schedule;

        $parcel_info = $this->parcelService->acceptParcel($request->parcel_id, $driver_login_id, $is_schedule);
        if ($parcel_info) {
            return apiResponse(1, 'Parcel accepted successfully.', $parcel_info, 200);
        }

        return apiResponse('1', 'Failed to accept parcel.', '', 400);
    }

    public function acceptAfterParcel(Request $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];


        $data = $this->parcelService->acceptAfterParcel($request->parcel_id, $driver_login_id);
        if ($data) {
            return apiResponse(1, 'Start Ride.', $data, 200);
        }

        return apiResponse('1', 'Failed to accept parcel.', '', 400);
    }

    public function inProcessParcel(Request $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];

        #$driver = Drivers::where('id', $driver_login_id)->first();

        $data = $this->parcelService->inProcessParcel($driver_login_id);
        if ($data) {
            return apiResponse(1, 'In Process Ride.', $data, 200);
        }

        return apiResponse('1', 'Failed to get parcel.', '', 400);
    }

    public function ArrivedPickupLocation(Request $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];


        $data = $this->parcelService->startParcel($request->parcel_id, $driver_login_id, DRIVER_ARRIVED);
        if($data['success']) {
            return apiResponse(1, $data['message'], $data['parcel'], 200);
        }

        return apiResponse('0', $data['message'], 'null', 400);
    }

    public function startParcel(Request $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];


        $data = $this->parcelService->startParcel($request->parcel_id, $driver_login_id, START_TRACK);
        if($data['success']) {
            return apiResponse(1, $data['message'], $data['parcel'], 200);
        }

        return apiResponse('0', $data['message'], 'null', 400);

    }

    public function trackParcel(Request $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];

        $trackParcel = $this->parcelService->trackParcel($request->parcel_id, $driver_login_id, [7]);
        if($trackParcel)
        return apiResponse(1, 'Track Parcel', $trackParcel, 200);
        else
        return apiResponse(0, 'There is no ride', [], 200);
    }

    public function cancelParcel(ParcelRequst $request)
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();
        $parcelId   = $validated['parcel_id'];
        $reason     = $validated['reason'];
        $driverId   = $login_detail['login_id'];

        $success = $this->parcelService->cancelParcel($parcelId, $driverId, $reason);

        if($success)
        return apiResponse(1, 'Parcel canceled successfully.', '', 200);

        return apiResponse(0, 'Parcel could not be canceled.', '', 400);

    }

    public function parcelSummary(ParcelRequst $request)
    {
        $validated  = $request->validated();
        $parcelId   = $validated['parcel_id'];

        $parcelSummary = $this->parcelService->getParcelSummary($parcelId);

        if(!$parcelSummary)
        return apiResponse(0, 'Parcel not found.', '', 404);

        return apiResponse(1, 'Parcel Summary.', $parcelSummary, 200);
    }

    public function pickupParcel(ParcelRequst $request) 
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();
        $parcelId   = $validated['parcel_id'];
        $otp        = $validated['otp'];  

        $parcel = $this->parcelService->pickupParcel($login_detail['login_id'], $parcelId, $otp); 

        if(is_string($parcel))
        return apiResponse(0, $parcel, '', 400);   
        
        return apiResponse(1, 'Parcel successfully picked up', $parcel, 200);
    }

    public function deliverParcel(ParcelRequst $request) 
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();
        $parcelId   = $validated['parcel_id'];
        $otp        = '1234'; //$validated['otp'];  

        $parcel = $this->parcelService->deliverParcel($login_detail['login_id'], $parcelId, $otp); 

        if(is_string($parcel))
        return apiResponse(0, $parcel, '', 400);   
        
        return apiResponse(1, 'Parcel successfully delivered', $parcel, 200);
    }

    public function historyParcel(Request $request)
    {
        $login_detail= $request->login_detail;
    
        $parcel = $this->parcelService->historyParcel($login_detail['login_id']);

        if($parcel)
        return apiResponse(1, 'Parcel Delivered List', $parcel, 200);
        
        return apiResponse(0, 'There is no data', '', 200); 
    }

    public function paymentReceived(ParcelRequst $request)
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();
        
        $parcel = $this->parcelService->updatePaymentStatus($login_detail['login_id'], $validated['parcel_id']);

        return apiResponse(1, 'Payment Received successfully.', $parcel, 200);
        
    }

    public function dashboardStats(Request $request)
    {
        $driverId = $request->login_detail['login_id'];
        $today = Carbon::today()->toDateString();

        $data = Parcels::selectRaw("
            SUM(final_price) as total_earnings,
            SUM(CASE WHEN DATE(created_at) = '$today' THEN final_price ELSE 0 END) as today_earnings,
            COUNT(*) as total_rides
        ")
    ->where('driver_id', $driverId)
    ->where('status', COMPLETE)
    ->first();

        $data->rating = 4;
        return apiResponse(1, '', $data, 200);
         
    }

    public function driverOnTheWay(ParcelRequst $request)
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();

        $parcel = $this->parcelService->onTheWay($validated['parcel_id'], $login_detail['login_id'], ON_THE_WAY);
        if($parcel)
        return apiResponse(1, 'On The Way.', $parcel, 200); 
        else
        return apiResponse(0, 'Something went wrong.', [], 200);
       
    } 
    

}

