<?php

namespace App\Repositories;

use App\Models\Drivers;
use App\Models\DriverEarning;


class DriverRepository
{
    /**
     * Save file information and vehicle details to the driver model.
     *
     * @param Driver $driver
     * @param array $filePaths
     * @param array $vehicleData
     * @return Driver
     */
    public function save(Drivers $driver, array $filePaths, array $vehicleData = [], $id): Drivers
    {
        $driver_obj     = auth('api_driver')->user();
        $driver_obj->update([
            'registration_certificate'  => $filePaths['registration_certificate'] ?? null,
            'vehicle_insurance'         => $filePaths['vehicle_insurance'] ?? null,
            'driving_licence'           => $filePaths['driving_licence'] ?? null,
            'profile_photo'             => $filePaths['profile_photo'] ?? null,
            'brand'                     => $vehicleData['brand'],
            'model'                     => $vehicleData['model'],
            'year_of_manufacture'       => $vehicleData['year_of_manufacture'],
            'vehicle_type'              => $vehicleData['vehicle_type'],
            'registration_number'       => $vehicleData['registration_number'],
            'vehicle_color'             => $vehicleData['vehicle_color'],
            'vehicle_photo'             => $filePaths['vehicle_photo'] 
        ]);

        return $driver_obj;
    }

    protected function handleFileUploads(Request $request): array
    {
        $filePaths = [];

        // Check and store each file if present
        if ($request->hasFile('rc_file')) {
            $filePaths['rc_file'] = $request->file('rc_file')->store('uploads/rc_files', 'public');
        }

        if ($request->hasFile('dl_file')) {
            $filePaths['dl_file'] = $request->file('dl_file')->store('uploads/dl_files', 'public');
        }

        if ($request->hasFile('profile_photo')) {
            $filePaths['profile_photo'] = $request->file('profile_photo')->store('uploads/profile_photos', 'public');
        }

        return $filePaths;
    }

    public function find($id)
    {
        return Drivers::findOrFail($id);
    } 

    public function updateDLInfo($driver, $dlFrontPath = null, $dlBackPath = null)
    {
        $driver = Drivers::find($driver['login_id']);
            
        if ($dlFrontPath || $dlBackPath) {
            $dlPath = implode(',', array_filter([$dlFrontPath, $dlBackPath]));
            $driver->driving_licence = $dlPath;
            $driver->save();            
        }
        return $driver;
    }

    public function updateRCInfo($driver, $rcPath)
    {
        $driver = Drivers::find($driver['login_id']);
            
        if ($rcPath) {
            $driver->registration_certificate = $rcPath;
            $driver->save();            
        }
        return $driver;
    }

    public function updateInsuranceInfo($driver, $insurancePath)
    {
        $driver = Drivers::find($driver['login_id']);
            
        if ($insurancePath) {
            $driver->vehicle_insurance = $insurancePath;
            $driver->save();            
        }
        return $driver;
    }

    public function update(Drivers $driver, array $data)
    {
        return $driver->update($data);
    }

    public function updateTermsLegal(int $driverId, $status, $updateType)
    {
        $driver = Drivers::find($driverId);
        if($updateType == 'terms')
        $driver->terms = $status;
        else if($updateType == 'legal')
        $driver->legal = $status;        

        $driver->save();

        return $driver;
    }

    public function updateLatLongAsPicup(int $driverId, $data)
    {
        $driver = Drivers::find($driverId);
        $driver->latitude   = $data['pickup_latitude'];        
        $driver->longitude  = $data['pickup_longitude'];        

        $driver->save();

        return $driver;
    }

    public function storeDriverEarnings(int $driverId, int $parcelId, float $totalPrice, float $driverCommission, float $distanceCovered)
    {
        return DriverEarning::create([
            'driver_id'         => $driverId,
            'parcel_id'         => $parcelId,
            'total_price'       => sprintf('%.2f',$totalPrice),
            'driver_commission' => sprintf('%.2f',$driverCommission),
            'distance_covered'  => sprintf('%.2f',$distanceCovered),
        ]);
    }

    public function earning(int $driverId)
    {
        return DriverEarning::with('parcel')->find($driverId);    
    }  

    public function isProfileComplete(int $driverId)
    {
        $driver = Drivers::find($driverId);
        return $driver->profile_photo && $driver->vehicle_insurance && $driver->vehicle_photo && $driver->driving_licence && $driver->registration_certificate;       
    }

    public function updateProfileCompleteStatus(int $driverId, $status)
    {
        $driver = Drivers::find($driverId);
        $driver->profile_complete = $status;
        $driver->save();

        return $driver;
    }   

}
