<?php

namespace App\Repositories;

use App\Models\Parcels;
use App\Models\ParcelRequest;

use App\Models\Drivers;
use App\Models\Category;
use App\Models\Discount;

use Carbon\Carbon;

use Illuminate\Support\Facades\DB;

class ParcelRepository
{
    //////////USER//////////
    public function createParcel(array $data)
    {
        $this->driverUpdateCoordinate($data['pickup_latitude'], $data['pickup_longitude']);
        return Parcels::create($data);
    }

    public function driverUpdateCoordinate($customerLat, $customerLong)
    {
        $rangeKm = 2;
        $newCoor= generateRandomLocation(floatval($customerLat), floatval($customerLong), $rangeKm);
        
        Drivers::where('status', '1')->update([
            'latitude' => $newCoor['lat'],
            'longitude' => $newCoor['lng']
        ]);

    }

    public function isAcceptRide()
    {
        return Parcels::where('status', ASSIGNED)
        ->with('user')
        ->with('driver')
        ->latest()
        ->first();
    }

    public function inProcessRide($userId)
    {
       return Parcels::where([
            'user_id' => $userId
            ])
        ->whereIn('status', ['1', '2', '3', '7'])
        ->with('user')
        ->with('driver')
        ->first();

    }

    public function startRideStatus($userId, $parcelId)
    {
       return Parcels::where([
            'user_id'   => $userId,
            'id'        => $parcelId
            ])
        ->where('status', '7')
        ->with('user')
        ->with('driver')
        ->first();

    }

    public function allRides($userId)
    {

        $categories = \App\Models\Category::where('status', '1')->get()->keyBy('id');
        return Parcels::with([
                'driver:id,category',
                'driver.category:id,name'
            ])
            ->where('user_id', $userId)
            ->select([
                'id',
                'parcel_type',
                'contact_name',
                'contact_mobile',
                'pickup_address',
                'delivery_address',
                'driver_id',
                'schedule_ride',
                'created_at',
                'status',
                'payment',
                'final_price'
            ])
            ->orderBy('id', 'desc')
            ->get()
            ->map(function ($parcel) use($categories){


                $categoryId = optional($parcel->driver);
                $category = $categories->get($categoryId['category']);
                $photo = optional($category)->photo;
                $vehicle_img = '';
                if($photo)
                {
                    $vehicle_root_url = url('').'/category/';
                    $vehicle_img = $vehicle_root_url.$photo;
                }

                return [
                    'id' => $parcel->id,
                    'contact_name' => $parcel->contact_name,
                    'contact_mobile' => $parcel->contact_mobile,
                    'from' => $parcel->pickup_address,
                    'to' => $parcel->delivery_address,
                    'driver_id'=> $parcel->driver_id,
                    'vehicle_image' => $vehicle_img,
                    'ride_type' => ride_type($parcel->parcel_type),
                    'created_at' => $parcel->schedule_ride ? $parcel->schedule_ride : $parcel->created_at->toDateTimeString(),
                    'schedule_ride' => $parcel->schedule_ride,
                    'status' => $parcel->status,
                    'screen_name'=>ScreenNameForUser($parcel->status),
                    'status_name' => statusName($parcel->status),
                    'payment' => $parcel->payment,
                    'final_price' => $parcel->final_price,
                    'currency' => currency_icon(),
                ];
            });
    }
     
    public function allCoupon()
    {
        return Discount::where('status', '1')
            ->orderBy('id', 'desc')
            ->get()
            ->map(function ($coupon){

                return [
                    'id'            => $coupon->id,
                    'name'          => $coupon->name,
                    'description'   => $coupon->description
                ];            

 
            });
    }

    public function cancelParcelByUser(int $parcelId, int $userId, string $reason): bool
    {
        return DB::transaction(function () use ($parcelId, $userId, $reason) {
            $parcelRequest = Parcels::where('id', $parcelId)
                                          ->where('user_id', $userId)
                                          ->first();

            if (!$parcelRequest || $parcelRequest->status !== '1') {
                return false;
            }

            $parcel = Parcels::find($parcelId);
            if ($parcel) {
                $parcel->driver_id = null;
                $parcel->status = CANCELED;
                $parcel->description = $reason;
                return $parcel->save();
            }

            return false;
        });
    } 

    ////////////////////

    public function getLatestParcel()
    {
        return Parcels::where('status', 1)
        ->with('user')
        ->latest()
        ->first();

    }

    public function acceptAfterParcel($parcelId, $driverId, $statuses = [2, 9])
    {
        /*
        return Parcels::where([
                'status'    => $status,
                //'id'        => $parcelId,
                'driver_id' => $driverId
        ])
        ->with(['user', 'driver'])
        ->first();*/

        return Parcels::whereIn('status', $statuses)
        ->where('driver_id', $driverId)
        ->with(['user', 'driver'])
        ->first();

    }

    public function inProcessParcel($driverId)
    {
       return Parcels::where('driver_id', $driverId)
        ->whereIn('status', [2, 3, 7, 9])
        ->with(['user', 'driver'])
        ->first();

    }
    

    public function getPendingParcels()
    {
        return Parcels::where('status', '1')->get();
    }

    public function getNearbyParcels($status, $latitude, $longitude, $radius)
    {
        return Parcels::select(
                'parcels.*',
                DB::raw("(
                    6371 * acos(
                        cos(radians($latitude)) * 
                        cos(radians(pickup_latitude)) * 
                        cos(radians(pickup_longitude) - radians($longitude)) + 
                        sin(radians($latitude)) * 
                        sin(radians(pickup_latitude))
                    )
                ) AS distance")
            )
            ->where('status', $status)
            ->having('distance', '<=', $radius)
            ->orderBy('id', 'DESC')
            ->get();
    }

    public function cancelParcel(int $parcelId, int $driverId, string $reason): bool
    {
        return DB::transaction(function () use ($parcelId, $driverId, $reason) {
            $parcelRequest = ParcelRequest::where('parcel_id', $parcelId)
                                          ->where('driver_id', $driverId)
                                          ->first();

            if (!$parcelRequest || $parcelRequest->status !== '1') {
                return false;
            }

            $parcelRequest->update([
                'status' => CANCELED,
                'cancellation_reason' => $reason,
            ]);

            $parcel = Parcels::find($parcelId);
            if ($parcel) {
                $parcel->driver_id = null;
                $parcel->status = '1';
                return $parcel->save();
            }

            return false;
        });
    }

    public function findById(int $parcelId)
    {
        return Parcels::with('user', 'driver')->find($parcelId);
    } 

    public function historyParcel($driverId)
    {
        $now = Carbon::now();
        $parcels = Parcels::query()
        ->select(['id', 'user_id', 'driver_id', 'status', 'delivery_time', 'final_price', 'pickup_address', 'delivery_address', 'schedule_ride', 'created_at'])
        ->where('driver_id', $driverId)
        ->whereIn('status', [4, 11])
        ->with(['user:id,name,email,mobile'])
        ->orderByRaw("
            CASE
                WHEN schedule_ride IS NOT NULL THEN schedule_ride
                ELSE created_at
            END DESC
        ")
        ->get()
        ->map(function ($parcel) use ($now) {
        $parcel->can_accept_ride = true;

        if (
            $parcel->status == 11 &&
            $parcel->schedule_ride &&
            $now->diffInMinutes(Carbon::parse($parcel->schedule_ride), false) <= 20 &&
            $now->diffInMinutes(Carbon::parse($parcel->schedule_ride), false) >= 0
        ) {
            $parcel->can_accept_ride = true;
        }

        return $parcel;
        });
        return $parcels;
 
    }

    public function updatePaymentStatus($driverId, $parcelId)
    {
        $parcel = Parcels::where('id', $parcelId)
                        ->where('driver_id', $driverId)
                        ->firstOrFail();
        $parcel->payment_received = '1';
        $parcel->save();
        
        return $parcel;
    }    
    
}
